import time
import argparse
from molactivity.E2_smiles_to_images import smiles_to_images

def main():
    parser = argparse.ArgumentParser(
        description='Generate images from SMILES strings',
        formatter_class=argparse.RawDescriptionHelpFormatter
    )
    parser.add_argument('-d', '--dataset', default='test', help='Dataset to use')
    parser.add_argument('-f', '--input_file', default='./smiles_to_image_sample.csv', 
                       help='Input CSV file containing SMILES data (default: ./predict_sample5.csv)')
    parser.add_argument('-o', '--output_dir', default='./ImageData', 
                       help='Output directory for generated images (default: ./ImageData5)')
    parser.add_argument('-lw', '--line_width', type=float, default=10,
                       help='Line width for molecular bonds (typical range: 1-50, default: 10)')
    parser.add_argument('-s', '--image_size', type=int, nargs=2, default=[2500, 2500],
                       help='Image size as width height (default: 2500 2500)')
    parser.add_argument('-fmt', '--image_format', default='png', 
                       choices=['png', 'jpg', 'jpeg', 'tif', 'tiff', 'svg'],
                       help='Output image format (default: png)')
    parser.add_argument('-max', '--max_atoms', type=int, default=149,
                       help='Maximum number of atoms to draw (default: 200, large peptides will be skipped)')
    args = parser.parse_args()
    
    print("Configuration:")
    print(f"  Input file: {args.input_file}")
    print(f"  Output directory: {args.output_dir}")
    print(f"  Max atoms limit: {args.max_atoms}")
    print(f"  Image size: {args.image_size[0]}x{args.image_size[1]}")
    print(f"  Image format: {args.image_format}")
    print(f"  Line width: {args.line_width}")
    
    start_time = time.time()
    smiles_to_images(
        dataset=args.dataset,
        input_file=args.input_file,
        output_dir=args.output_dir,
        line_width=args.line_width,
        image_size=tuple(args.image_size),
        image_format=args.image_format,
        max_atoms=args.max_atoms
    )
    end_time = time.time()
    print(f"\nTime used: {end_time - start_time:.2f} seconds")


if __name__ == "__main__":
    main()