import time


DATASET_FILE = 'train_sample.csv'

FAST_CONFIG = {
    'PARALLEL_EVALUATION': False,
    'MAX_WORKERS': None,
    'NUMPY_OPTIMIZATION': False,
    'SAVE_INDIVIDUAL_PREDICTIONS': True,
    
    'evaluation_parameters': {
        'device': 'cpu',
        'model_file': 'model_1.dict',
        'models': 'auto',
        'ensemble': True,
        'batch_size': 32,
        'fast_mode': True,
    },
    
    'optimization_parameters': {
        'use_vectorized_ops': True,
        'cache_fingerprints': True,
        'batch_processing': True,
        'memory_efficient': True,
    },
    
    'output_parameters': {
        'output_file': 'train_sample_with_predictions.csv',
        'analyze_quality': True,
        'fast_save': True,
    }
}

def apply_fast_configurations():
    try:
        from molactivity import B5_evaluate as evaluate
        
        def configured_fast_evaluation():
            config = FAST_CONFIG

            data_provider = evaluate.prepare_pure_predicting_dataset(
                DATASET_FILE, 
                fingerprint_type='Morgan', 
                batch_size=config['evaluation_parameters']['batch_size'], 
                shuffle=False
            )
            
            model_files = evaluate.parse_model_selection(config['evaluation_parameters']['models'])
            print(f"using {len(model_files)} models for fast evaluation")
            
            predictions = []
            std_devs = []
            individual_predictions = {}
            
            if config['PARALLEL_EVALUATION'] and len(model_files) > 1:
                print("using fast parallel evaluation")
                try:
                    from molactivity.B6_evaluate_fast_parallel import generate_fast_parallel_ensemble_predictions
                    predictions, std_devs, individual_predictions = generate_fast_parallel_ensemble_predictions(
                        model_files=model_files,
                        dataset_file=DATASET_FILE,
                        max_workers=config['MAX_WORKERS'],
                        batch_size=config['evaluation_parameters']['batch_size'],
                        optimization_config=config['optimization_parameters']
                    )
                    print(f"evaluation completed: {len(predictions)} predictions")
                    
                except ImportError as e:
                    print(f"fast parallel evaluation failed: {e}")
                    print("falling back to fast sequential evaluation...")
                    config['PARALLEL_EVALUATION'] = False
                except Exception as e:
                    print(f"fast parallel evaluation failed: {e}")
                    print("falling back to fast sequential evaluation...")
                    config['PARALLEL_EVALUATION'] = False
            
            if not config['PARALLEL_EVALUATION'] or len(model_files) == 1:
                
                loaded_models = evaluate.load_multiple_models(model_files)
                predictions, std_devs, individual_predictions = evaluate.generate_ensemble_predictions(loaded_models, data_provider, model_files=model_files)
            
            output_file = config['output_parameters']['output_file']
            if predictions:
                individual_preds_to_save = individual_predictions if config['SAVE_INDIVIDUAL_PREDICTIONS'] else None
                
                if config['evaluation_parameters']['ensemble'] and std_devs:
                    save_success = evaluate.save_predictions_to_csv(
                        predictions, output_file, std_devs, individual_preds_to_save, input_file=DATASET_FILE
                    )
                else:
                    save_success = evaluate.save_predictions_to_csv(
                        predictions, output_file, individual_predictions=individual_preds_to_save, input_file=DATASET_FILE
                    )
                
                if save_success and config['output_parameters']['analyze_quality']:
                    evaluate.analyze_prediction_quality(output_file)
                elif not save_success:
                    print("failed to save predictions")
            else:
                print("no predictions to save")
        
        evaluate.evaluation = configured_fast_evaluation
        
        return configured_fast_evaluation
        
    except ImportError as e:
        print(f"configuration failed: {e}")
        return None
    except Exception as e:
        print(f"configuration failed: {e}")
        return None

def run_fast_evaluation():
    config = FAST_CONFIG
    print("FAST_CONFIG:")
    print(f"device: {config['evaluation_parameters']['device']}")
    print(f"parallel evaluation: {config['PARALLEL_EVALUATION']}")
    print(f"max workers: {config['MAX_WORKERS']}")
    print(f"models: {config['evaluation_parameters']['models']}")
    print(f"batch size: {config['evaluation_parameters']['batch_size']}")
    print(f"dataset: {DATASET_FILE}")
    print(f"output file: {config['output_parameters']['output_file']}")
    
    configured_evaluation = apply_fast_configurations()
    
    if configured_evaluation:
        configured_evaluation()
    else:
        print("Configuration failed")

if __name__ == "__main__":
    start_time = time.time()
    
    run_fast_evaluation()
    
    end_time = time.time()
    total_time = end_time - start_time
    print(f"time used for fast evaluation: {total_time:.2f} seconds")